/**
 * タスクマネージャー - Background Service Worker
 *
 * 機能:
 * - タスクのCRUD操作
 * - バッジ更新（未完了タスク数）
 * - 通知・リマインダー
 * - chrome.storage.local でのデータ管理
 */

// ========== ユーティリティ関数 ==========

// UUID生成（kintone-task-saverパターン）
function generateUUID() {
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
    const r = Math.random() * 16 | 0;
    const v = c === 'x' ? r : (r & 0x3 | 0x8);
    return v.toString(16);
  });
}

// バッジ更新（kintone-task-saverパターン）
async function updateBadge() {
  try {
    const result = await chrome.storage.local.get(['tasks']);
    const tasks = result.tasks || [];
    const incompleteCount = tasks.filter(t => !t.completed).length;

    if (incompleteCount > 0) {
      chrome.action.setBadgeText({ text: incompleteCount.toString() });
      chrome.action.setBadgeBackgroundColor({ color: '#3b82f6' });
    } else {
      chrome.action.setBadgeText({ text: '' });
    }
  } catch (error) {
    console.error('Failed to update badge:', error);
  }
}

// ========== タスクCRUD関数 ==========

// タスク作成
async function createTask(taskData) {
  try {
    const result = await chrome.storage.local.get(['tasks', 'settings']);
    const tasks = result.tasks || [];
    const settings = result.settings || {};

    const newTask = {
      id: generateUUID(),
      title: taskData.title,
      description: taskData.description || '',
      completed: false,
      priority: taskData.priority || 'medium',
      deadline: taskData.deadline || null,
      category: taskData.category || null,
      tags: taskData.tags || [],
      order: tasks.length,
      progress: taskData.progress !== undefined ? taskData.progress : 0,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      completedAt: null
    };

    tasks.unshift(newTask);  // 先頭に追加（kintone-task-saverパターン）

    // 100件制限（kintone-task-saverパターン）
    const limitedTasks = tasks.slice(0, 100);

    await chrome.storage.local.set({ tasks: limitedTasks });
    await updateBadge();

    return newTask;
  } catch (error) {
    console.error('Failed to create task:', error);
    throw error;
  }
}

// タスク更新
async function updateTask(taskId, updates) {
  try {
    const result = await chrome.storage.local.get(['tasks']);
    const tasks = result.tasks || [];
    const index = tasks.findIndex(t => t.id === taskId);

    if (index === -1) {
      throw new Error('Task not found');
    }

    // 不変性を保つスプレッド構文（kintone-task-saverパターン）
    tasks[index] = {
      ...tasks[index],
      ...updates,
      updatedAt: new Date().toISOString()
    };

    // 完了状態の変更
    if (updates.completed !== undefined && updates.completed !== tasks[index].completed) {
      if (updates.completed) {
        tasks[index].completedAt = new Date().toISOString();
        tasks[index].progress = 100;
      } else {
        tasks[index].completedAt = null;
        tasks[index].progress = tasks[index].progress === 100 ? 0 : tasks[index].progress;
      }
    }

    await chrome.storage.local.set({ tasks });
    await updateBadge();

    return tasks[index];
  } catch (error) {
    console.error('Failed to update task:', error);
    throw error;
  }
}

// タスク削除
async function deleteTask(taskId) {
  try {
    const result = await chrome.storage.local.get(['tasks']);
    const tasks = result.tasks || [];
    const filteredTasks = tasks.filter(t => t.id !== taskId);

    await chrome.storage.local.set({ tasks: filteredTasks });
    await updateBadge();

    return { success: true };
  } catch (error) {
    console.error('Failed to delete task:', error);
    throw error;
  }
}

// ========== メッセージハンドラー ==========

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  // タスクリスト取得
  if (request.action === 'getTasks') {
    chrome.storage.local.get(['tasks']).then(result => {
      sendResponse({ success: true, tasks: result.tasks || [] });
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true; // 非同期レスポンス
  }

  // タスク作成
  if (request.action === 'createTask') {
    createTask(request.taskData).then(task => {
      sendResponse({ success: true, task });
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true;
  }

  // タスクの完了/未完了切り替え（kintone-task-saverパターン）
  if (request.action === 'toggleTask') {
    chrome.storage.local.get(['tasks']).then(result => {
      const tasks = result.tasks || [];
      const taskIndex = tasks.findIndex(t => t.id === request.taskId);

      if (taskIndex !== -1) {
        const updatedTasks = [...tasks];
        updatedTasks[taskIndex] = {
          ...updatedTasks[taskIndex],
          completed: !updatedTasks[taskIndex].completed
        };

        chrome.storage.local.set({ tasks: updatedTasks }).then(() => {
          updateBadge();
          sendResponse({ success: true, task: updatedTasks[taskIndex] });
        });
      } else {
        sendResponse({ success: false, error: 'Task not found' });
      }
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true;
  }

  // タスク削除
  if (request.action === 'deleteTask') {
    deleteTask(request.taskId).then(() => {
      sendResponse({ success: true });
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true;
  }

  // タスク編集
  if (request.action === 'updateTaskText') {
    chrome.storage.local.get(['tasks']).then(result => {
      const tasks = result.tasks || [];
      const taskIndex = tasks.findIndex(t => t.id === request.taskId);

      if (taskIndex !== -1) {
        const updatedTasks = [...tasks];
        updatedTasks[taskIndex] = {
          ...updatedTasks[taskIndex],
          title: request.text,
          updatedAt: new Date().toISOString()
        };

        chrome.storage.local.set({ tasks: updatedTasks }).then(() => {
          sendResponse({ success: true, task: updatedTasks[taskIndex] });
        });
      } else {
        sendResponse({ success: false, error: 'Task not found' });
      }
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true;
  }

  // 完了済みタスクをクリア
  if (request.action === 'clearCompleted') {
    chrome.storage.local.get(['tasks']).then(result => {
      const tasks = result.tasks || [];
      const incompleteTasks = tasks.filter(t => !t.completed);

      chrome.storage.local.set({ tasks: incompleteTasks }).then(() => {
        updateBadge();
        sendResponse({ success: true, count: tasks.length - incompleteTasks.length });
      });
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true;
  }

  // すべて削除
  if (request.action === 'clearAll') {
    chrome.storage.local.set({ tasks: [] }).then(() => {
      updateBadge();
      sendResponse({ success: true });
    });
    return true;
  }

  // タスク順序更新
  if (request.action === 'updateTaskOrder') {
    chrome.storage.local.get(['tasks']).then(result => {
      const tasks = result.tasks || [];

      // 新しい順序を適用
      request.newOrder.forEach(item => {
        const task = tasks.find(t => t.id === item.id);
        if (task) {
          task.order = item.order;
        }
      });

      // 順序でソート
      tasks.sort((a, b) => a.order - b.order);

      chrome.storage.local.set({ tasks }).then(() => {
        sendResponse({ success: true });
      });
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true;
  }
});

// ========== インストール時・アラーム ==========

// 拡張機能インストール時
chrome.runtime.onInstalled.addListener(async () => {
  // 既存データを確認
  const result = await chrome.storage.local.get(['tasks', 'categories', 'tags', 'settings']);

  // 初期データが存在しない場合のみ設定（初回インストール時のみ）
  if (!result.tasks) {
    await chrome.storage.local.set({
      tasks: [],
      categories: [
        { id: 'category-1', name: '仕事', color: '#3b82f6', order: 0 },
        { id: 'category-2', name: '個人', color: '#10b981', order: 1 }
      ],
      tags: [
        { id: 'tag-1', name: '重要', color: '#ef4444' },
        { id: 'tag-2', name: '急ぎ', color: '#f59e0b' }
      ],
      settings: {
        theme: 'light',
        defaultCategory: '仕事',
        reminderEnabled: true,
        reminderTime: '09:00',
        completedTasksVisibility: 'hide'
      }
    });
  }

  // 初期化時にバッジ更新
  await updateBadge();

  // コンテキストメニュー作成
  chrome.contextMenus.create({
    id: 'saveAsTask',
    title: 'タスクとして保存',
    contexts: ['selection']
  });

  // アラーム作成（日次リマインダー）
  chrome.alarms.create('dailyReminder', {
    delayInMinutes: 1,
    periodInMinutes: 1440  // 24時間ごと
  });
});

// アラーム発火時の処理
chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name === 'dailyReminder') {
    await sendDailyReminder();
  }
});

// 日次リマインダー送信
async function sendDailyReminder() {
  try {
    const result = await chrome.storage.local.get(['tasks', 'settings']);
    const tasks = result.tasks || [];
    const settings = result.settings || {};

    if (!settings.reminderEnabled) return;

    const today = new Date().toISOString().split('T')[0];
    const dueTodayTasks = tasks.filter(t =>
      !t.completed &&
      t.deadline === today
    );

    if (dueTodayTasks.length === 0) return;

    // 通知作成（kintone-task-saverパターン）
    await chrome.notifications.create({
      type: 'basic',
      iconUrl: chrome.runtime.getURL('icons/icon48.png'),
      title: 'タスクマネージャー',
      message: `本日期限のタスクが${dueTodayTasks.length}件あります`,
      buttons: [
        { title: 'サイドパネルを開く' }
      ]
    });
  } catch (error) {
    console.error('Failed to send daily reminder:', error);
  }
}

// 通知ボタンクリック時の処理
chrome.notifications.onButtonClicked.addListener(async (notificationId, buttonIndex) => {
  if (buttonIndex === 0) {
    // サイドパネルを開く
    const windows = await chrome.windows.getAll({ populate: false });
    if (windows.length > 0) {
      await chrome.sidePanel.open({ windowId: windows[0].id });
    }
  }
});

// コンテキストメニュークリック時の処理
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'saveAsTask') {
    const selectedText = info.selectionText?.trim();

    if (!selectedText) {
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: chrome.runtime.getURL('icons/icon48.png'),
        title: 'タスクマネージャー',
        message: 'テキストが選択されていません'
      });
      return;
    }

    // タスクデータ作成
    const taskData = {
      title: selectedText,
      priority: 'medium',
      deadline: null,
      category: null,
      tags: []
    };

    try {
      // タスクを保存
      const newTask = await createTask(taskData);

      // 通知
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: chrome.runtime.getURL('icons/icon48.png'),
        title: 'タスクマネージャー',
        message: 'タスクを追加しました',
        buttons: [
          { title: 'サイドパネルを開く' }
        ]
      });
    } catch (error) {
      console.error('Failed to save task:', error);
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: chrome.runtime.getURL('icons/icon48.png'),
        title: 'タスクマネージャー',
        message: 'タスクの追加に失敗しました'
      });
    }
  }
});
