/**
 * タスクマネージャー - Sidepanel Script
 *
 * 機能:
 * - タスク一覧の表示
 * - タスクの追加・完了・削除・編集
 * - フィルタリング
 * - ドラッグ&ドロップによる並び替え
 * - テーマ切り替え
 * - 統計表示
 *
 * 参照パターン: kintone-task-saver/src/popup/popup.js
 */

// ========== DOM要素 ==========
const taskListEl = document.getElementById('task-list');
const totalCountEl = document.getElementById('total-count');
const incompleteCountEl = document.getElementById('incomplete-count');
const taskForm = document.getElementById('task-form');
const taskInput = document.getElementById('task-input');
const prioritySelect = document.getElementById('priority-select');
const deadlineInput = document.getElementById('deadline-input');
const categorySelect = document.getElementById('category-select');
const tagsInput = document.getElementById('tags-input');
const progressInput = document.getElementById('progress-input');
const progressValue = document.getElementById('progress-value');
const themeToggleBtn = document.getElementById('theme-toggle-btn');
const statsToggleBtn = document.getElementById('stats-toggle-btn');
const statisticsPanel = document.getElementById('statistics-panel');
const tagFilter = document.getElementById('tag-filter');
const sortBtn = document.getElementById('sort-btn');
const sortLabel = document.getElementById('sort-label');

// ========== 状態（kintone-task-saverパターン） ==========
let tasks = [];
let currentStatusFilter = 'all';  // 現在のステータスフィルター
let currentTagFilter = '';  // 現在のタグフィルター
let currentSortOrder = 'default';  // 現在のソート順（default/deadline-asc/deadline-desc）
let editingTaskId = null;  // 編集中のタスクID
let categories = [];
let settings = {};

// ========== 初期化 ==========
document.addEventListener('DOMContentLoaded', () => {
  loadTasks();
  loadCategories();
  loadSettings();
  setupEventListeners();
  setupDragAndDrop();
});

// ========== イベントリスナー設定 ==========
function setupEventListeners() {
  // タスク追加フォーム
  taskForm.addEventListener('submit', handleAddTask);

  // ステータスフィルターボタン（kintone-task-saverパターン）
  document.querySelectorAll('.status-filter-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      document.querySelectorAll('.status-filter-btn').forEach(b => b.classList.remove('active'));
      btn.classList.add('active');
      currentStatusFilter = btn.dataset.filter;
      renderTasks();
    });
  });

  // テーマ切り替えボタン
  themeToggleBtn.addEventListener('click', toggleTheme);

  // 統計表示ボタン
  statsToggleBtn.addEventListener('click', toggleStatistics);

  // 進捗スライダー
  progressInput.addEventListener('input', (e) => {
    progressValue.textContent = e.target.value;
  });

  // タグフィルター
  tagFilter.addEventListener('change', (e) => {
    currentTagFilter = e.target.value;
    renderTasks();
  });

  // 期日ソートボタン
  sortBtn.addEventListener('click', () => {
    // ソート順を循環: default → deadline-asc → deadline-desc → default
    if (currentSortOrder === 'default') {
      currentSortOrder = 'deadline-asc';
      sortLabel.textContent = '期日▼';
    } else if (currentSortOrder === 'deadline-asc') {
      currentSortOrder = 'deadline-desc';
      sortLabel.textContent = '期日▲';
    } else {
      currentSortOrder = 'default';
      sortLabel.textContent = 'デフォルト';
    }

    // アクティブスタイルの切り替え
    if (currentSortOrder === 'default') {
      sortBtn.classList.remove('active');
    } else {
      sortBtn.classList.add('active');
    }

    renderTasks();
  });
}

// ========== ドラッグ&ドロップセットアップ ==========
function setupDragAndDrop() {
  let draggedTaskId = null;
  let draggedElement = null;

  taskListEl.addEventListener('dragstart', (e) => {
    const taskItem = e.target.closest('.task-item');

    if (!taskItem) return;

    draggedTaskId = taskItem.dataset.taskId;
    draggedElement = taskItem;

    taskItem.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
  });

  taskListEl.addEventListener('dragover', (e) => {
    e.preventDefault();

    const taskItem = e.target.closest('.task-item');

    if (!taskItem || taskItem.dataset.taskId === draggedTaskId) return;

    const rect = taskItem.getBoundingClientRect();
    const midpoint = rect.top + rect.height / 2;

    if (e.clientY < midpoint) {
      taskListEl.insertBefore(draggedElement, taskItem);
    } else {
      taskListEl.insertBefore(draggedElement, taskItem.nextSibling);
    }
  });

  taskListEl.addEventListener('drop', async (e) => {
    e.preventDefault();

    const taskItem = e.target.closest('.task-item');
    if (!taskItem) return;

    // 新しい順序を計算
    const newOrder = Array.from(taskListEl.children)
      .map((item, index) => ({
        id: item.dataset.taskId,
        order: index
      }));

    // 順序を保存
    try {
      const response = await chrome.runtime.sendMessage({
        action: 'updateTaskOrder',
        newOrder: newOrder
      });

      if (response.success) {
        // ローカルのタスクを更新
        newOrder.forEach(item => {
          const task = tasks.find(t => t.id === item.id);
          if (task) {
            task.order = item.order;
          }
        });
      }
    } catch (error) {
      console.error('Failed to update order:', error);
    }
  });

  taskListEl.addEventListener('dragend', (e) => {
    const taskItem = e.target.closest('.task-item');

    if (taskItem) {
      taskItem.classList.remove('dragging');
    }

    draggedTaskId = null;
    draggedElement = null;
  });
}

// ========== データ読み込み ==========

// タスク読み込み
async function loadTasks() {
  try {
    const response = await chrome.runtime.sendMessage({ action: 'getTasks' });
    if (!response.success) {
      throw new Error(response.error || 'タスクの読み込みに失敗しました');
    }
    tasks = response.tasks || [];
    renderTasks();
  } catch (error) {
    console.error('Failed to load tasks:', error);
    renderErrorState(error.message);
  }
}

// カテゴリ読み込み
async function loadCategories() {
  try {
    const result = await chrome.storage.local.get(['categories']);
    categories = result.categories || [];
    renderCategorySelect();
  } catch (error) {
    console.error('Failed to load categories:', error);
  }
}

// 設定読み込み
async function loadSettings() {
  try {
    const result = await chrome.storage.local.get(['settings']);
    settings = result.settings || {};

    // テーマ適用
    if (settings.theme) {
      document.documentElement.setAttribute('data-theme', settings.theme);
      updateThemeIcon(settings.theme);
    }
  } catch (error) {
    console.error('Failed to load settings:', error);
  }
}

// ========== レンダリング関数 ==========

// カテゴリセレクト描画
function renderCategorySelect() {
  let html = '<option value="">カテゴリなし</option>';
  categories.forEach(cat => {
    html += `<option value="${escapeHtml(cat.name)}">${escapeHtml(cat.name)}</option>`;
  });
  categorySelect.innerHTML = html;
}

// タグフィルター選択肢を更新
function updateTagFilterOptions() {
  // 全タスクから一意なタグを収集
  const allTags = new Set();
  tasks.forEach(task => {
    if (task.tags && Array.isArray(task.tags)) {
      task.tags.forEach(tag => {
        if (tag) allTags.add(tag);
      });
    }
  });

  // 現在の選択を保持
  const currentValue = tagFilter.value;

  // 選択肢を生成
  let html = '<option value="">すべてのタグ</option>';
  Array.from(allTags).sort().forEach(tag => {
    const selected = tag === currentValue ? ' selected' : '';
    html += `<option value="${escapeHtml(tag)}"${selected}>${escapeHtml(tag)}</option>`;
  });

  tagFilter.innerHTML = html;
}

// タスク描画
function renderTasks() {
  // ステータスフィルター適用
  let filteredTasks = tasks;
  if (currentStatusFilter === 'incomplete') {
    filteredTasks = tasks.filter(t => !t.completed);
  } else if (currentStatusFilter === 'completed') {
    filteredTasks = tasks.filter(t => t.completed);
  }

  // タグフィルター適用
  if (currentTagFilter) {
    filteredTasks = filteredTasks.filter(t =>
      t.tags && t.tags.includes(currentTagFilter)
    );
  }

  // 期日ソート適用
  if (currentSortOrder === 'deadline-asc') {
    // 期日昇順（近い順）、期日なしは最後
    filteredTasks.sort((a, b) => {
      if (!a.deadline && !b.deadline) return 0;
      if (!a.deadline) return 1;
      if (!b.deadline) return -1;
      return new Date(a.deadline) - new Date(b.deadline);
    });
  } else if (currentSortOrder === 'deadline-desc') {
    // 期日降順（遠い順）、期日なしは最後
    filteredTasks.sort((a, b) => {
      if (!a.deadline && !b.deadline) return 0;
      if (!a.deadline) return 1;
      if (!b.deadline) return -1;
      return new Date(b.deadline) - new Date(a.deadline);
    });
  }
  // default: 並び替えなし（作成日順）

  // カウント更新
  const totalCount = tasks.length;
  const incompleteCount = tasks.filter(t => !t.completed).length;
  totalCountEl.textContent = totalCount;
  incompleteCountEl.textContent = incompleteCount;

  // タグフィルター選択肢を更新
  updateTagFilterOptions();

  // 空状態チェック
  if (filteredTasks.length === 0) {
    renderEmptyState();
    return;
  }

  // タスクリスト描画
  const html = filteredTasks.map(task => renderTaskItem(task)).join('');
  taskListEl.innerHTML = html;

  // イベントリスナー設定
  setupTaskListeners();
}

// タスクアイテム描画
function renderTaskItem(task) {
  const isEditing = editingTaskId === task.id;

  // メタ情報
  const priorityLabel = {
    high: '高',
    medium: '中',
    low: '低'
  }[task.priority] || '';

  const metaHtml = [];

  if (priorityLabel) {
    metaHtml.push(`<span class="task-priority priority-${task.priority}">${escapeHtml(priorityLabel)}</span>`);
  }

  if (task.deadline) {
    const deadline = new Date(task.deadline).toLocaleDateString('ja-JP', {
      month: 'short',
      day: 'numeric'
    });
    metaHtml.push(`<span class="task-deadline">📅 ${escapeHtml(deadline)}</span>`);
  }

  if (task.category) {
    metaHtml.push(`<span class="task-category">${escapeHtml(task.category)}</span>`);
  }

  if (task.tags && task.tags.length > 0) {
    metaHtml.push(`<div class="task-tags">`);
    task.tags.forEach(tag => {
      metaHtml.push(`<span class="task-tag">#${escapeHtml(tag)}</span>`);
    });
    metaHtml.push(`</div>`);
  }

  // 進捗バー（すべての進捗値で表示）
  if (task.progress > 0) {
    metaHtml.push(`
      <div class="task-progress">
        <div class="progress-bar">
          <div class="progress-fill" style="width: ${task.progress}%"></div>
        </div>
        <span class="progress-text">${task.progress}%</span>
      </div>
    `);
  }

  // 編集モードの場合
  if (isEditing) {
    return `
      <div class="task-item editing" data-task-id="${task.id}">
        <div class="task-header">
          <div class="task-content edit-mode">
            <input
              type="text"
              class="task-edit-input"
              value="${escapeHtml(task.title)}"
              data-task-id="${task.id}"
              placeholder="タスク名を入力..."
            >
            <div class="edit-actions">
              <button class="save-edit-btn" data-task-id="${task.id}" title="保存">✓</button>
              <button class="cancel-edit-btn" title="キャンセル">✕</button>
            </div>
          </div>
        </div>
      </div>
    `;
  }

  // 通常表示
  return `
    <div class="task-item ${task.completed ? 'completed' : ''}" data-task-id="${task.id}" draggable="true">
      <div class="task-header">
        <div class="task-drag-handle" title="ドラッグして並び替え">⋮⋮</div>
        <input
          type="checkbox"
          class="task-checkbox"
          ${task.completed ? 'checked' : ''}
        >
        <div class="task-content">
          <div class="task-title" title="クリックして編集">${escapeHtml(task.title)}</div>
          <div class="task-meta">
            ${metaHtml.join('')}
          </div>
        </div>
        <div class="task-actions">
          <button class="edit-btn" data-task-id="${task.id}" title="編集">✎</button>
          <button class="delete-btn" data-task-id="${task.id}" title="削除">✕</button>
        </div>
      </div>
    </div>
  `;
}

// タスクイベントリスナー設定
function setupTaskListeners() {
  // チェックボックス
  document.querySelectorAll('.task-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', (e) => {
      const taskItem = e.target.closest('.task-item');
      const taskId = taskItem.dataset.taskId;
      toggleTask(taskId);
    });
  });

  // 削除ボタン
  document.querySelectorAll('.delete-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      const taskId = e.target.dataset.taskId;
      deleteTask(taskId);
    });
  });

  // 編集ボタン
  document.querySelectorAll('.edit-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      const taskId = e.target.dataset.taskId;
      startEditing(taskId);
    });
  });

  // タスクテキストをクリックで編集開始
  document.querySelectorAll('.task-title').forEach(textEl => {
    textEl.addEventListener('click', (e) => {
      const taskItem = e.target.closest('.task-item');
      const taskId = taskItem.dataset.taskId;
      startEditing(taskId);
    });
  });

  // 編集入力フィールド
  document.querySelectorAll('.task-edit-input').forEach(input => {
    const taskId = input.dataset.taskId;

    // Enterで保存、Escでキャンセル
    input.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        e.preventDefault();
        saveEdit(taskId, input.value);
      } else if (e.key === 'Escape') {
        e.preventDefault();
        cancelEdit();
      }
    });

    // フォーカス時に全選択
    input.addEventListener('focus', () => {
      input.select();
    });
  });

  // 保存ボタン
  document.querySelectorAll('.save-edit-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      const taskId = e.target.dataset.taskId;
      const input = document.querySelector(`.task-edit-input[data-task-id="${taskId}"]`);
      if (input) {
        saveEdit(taskId, input.value);
      }
    });
  });

  // キャンセルボタン
  document.querySelectorAll('.cancel-edit-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      cancelEdit();
    });
  });
}

// ========== タスクCRUD操作 ==========

// タスク追加
async function handleAddTask(e) {
  e.preventDefault();

  const title = taskInput.value.trim();
  if (!title) {
    alert('タスク名を入力してください');
    return;
  }

  // タグをカンマ区切りで解析
  const tagsValue = tagsInput.value.trim();
  const tags = tagsValue ? tagsValue.split(',').map(tag => tag.trim()).filter(tag => tag) : [];

  // 進捗を取得
  const progress = parseInt(progressInput.value, 10);

  const taskData = {
    title,
    priority: prioritySelect.value,
    deadline: deadlineInput.value || null,
    category: categorySelect.value || null,
    tags: tags,
    progress: progress
  };

  try {
    const response = await chrome.runtime.sendMessage({
      action: 'createTask',
      taskData
    });

    if (response.success) {
      tasks.unshift(response.task);
      renderTasks();

      // フォームリセット
      taskInput.value = '';
      prioritySelect.value = 'medium';
      deadlineInput.value = '';
      categorySelect.value = '';
      tagsInput.value = '';
      progressInput.value = '0';
      progressValue.textContent = '0';
      taskInput.focus();
    } else {
      alert(`エラー: ${response.error || 'タスクの追加に失敗しました'}`);
    }
  } catch (error) {
    console.error('Failed to add task:', error);
    alert(`エラーが発生しました: ${error.message}`);
  }
}

// タスクの完了/未完了切り替え（kintone-task-saverパターン）
async function toggleTask(taskId) {
  try {
    const response = await chrome.runtime.sendMessage({
      action: 'toggleTask',
      taskId: taskId
    });

    if (response.success) {
      const task = tasks.find(t => t.id === taskId);
      if (task) {
        task.completed = response.task.completed;
      }
      renderTasks();
    } else {
      alert(`エラー: ${response.error || 'タスクの更新に失敗しました'}`);
    }
  } catch (error) {
    console.error('Failed to toggle task:', error);
    alert(`エラーが発生しました: ${error.message}`);
  }
}

// タスク削除
async function deleteTask(taskId) {
  try {
    const response = await chrome.runtime.sendMessage({
      action: 'deleteTask',
      taskId: taskId
    });

    if (response.success) {
      tasks = tasks.filter(t => t.id !== taskId);
      renderTasks();
    } else {
      alert(`エラー: ${response.error || 'タスクの削除に失敗しました'}`);
    }
  } catch (error) {
    console.error('Failed to delete task:', error);
    alert(`エラーが発生しました: ${error.message}`);
  }
}

// ========== 編集機能（kintone-task-saverパターン） ==========

// 編集モード開始
function startEditing(taskId) {
  editingTaskId = taskId;
  renderTasks();

  // 入力フィールドにフォーカス
  setTimeout(() => {
    const input = document.querySelector(`.task-edit-input[data-task-id="${taskId}"]`);
    if (input) {
      input.focus();
      input.select();
    }
  }, 0);
}

// 編集キャンセル
function cancelEdit() {
  editingTaskId = null;
  renderTasks();
}

// 編集保存
async function saveEdit(taskId, newText) {
  const trimmedText = newText.trim();

  if (!trimmedText) {
    alert('タスク名を入力してください');
    return;
  }

  try {
    const response = await chrome.runtime.sendMessage({
      action: 'updateTaskText',
      taskId: taskId,
      text: trimmedText
    });

    if (response.success) {
      const task = tasks.find(t => t.id === taskId);
      if (task) {
        task.title = trimmedText;
      }
      editingTaskId = null;
      renderTasks();
    } else {
      alert(`エラー: ${response.error || 'タスクの更新に失敗しました'}`);
    }
  } catch (error) {
    console.error('Failed to update task:', error);
    alert(`エラーが発生しました: ${error.message}`);
  }
}

// ========== テーマ切り替え ==========

async function toggleTheme() {
  const currentTheme = document.documentElement.getAttribute('data-theme');
  const newTheme = currentTheme === 'dark' ? 'light' : 'dark';

  // テーマを適用
  document.documentElement.setAttribute('data-theme', newTheme);

  // 設定を保存
  try {
    await chrome.storage.local.set({
      settings: { ...settings, theme: newTheme }
    });
    settings.theme = newTheme;
    updateThemeIcon(newTheme);
  } catch (error) {
    console.error('Failed to save theme:', error);
  }
}

function updateThemeIcon(theme) {
  if (themeToggleBtn) {
    themeToggleBtn.textContent = theme === 'dark' ? '☀️' : '🌓';
  }
}

// ========== 統計表示 ==========

async function toggleStatistics() {
  const isHidden = statisticsPanel.classList.contains('hidden');

  if (isHidden) {
    // 統計を表示
    statisticsPanel.classList.remove('hidden');
    await renderStatistics();
  } else {
    // 統計を非表示
    statisticsPanel.classList.add('hidden');
  }
}

async function renderStatistics() {
  if (typeof Chart === 'undefined') {
    console.error('[Statistics] Chart.js is not loaded');
    alert('統計機能を読み込めませんでした。インターネット接続を確認して、拡張機能を再読み込みしてください。');
    return;
  }

  // 既存のチャートを破棄
  const existingCharts = Chart.getChart('completion-chart');
  if (existingCharts) {
    existingCharts.destroy();
  }

  const existingCategoryChart = Chart.getChart('category-chart');
  if (existingCategoryChart) {
    existingCategoryChart.destroy();
  }

  // 統計計算
  const totalTasks = tasks.length;
  const completedTasks = tasks.filter(t => t.completed).length;
  const incompleteTasks = totalTasks - completedTasks;

  // 全体の完了率（円グラフ）
  new Chart(document.getElementById('completion-chart'), {
    type: 'pie',
    data: {
      labels: ['完了', '未完了'],
      datasets: [{
        data: [completedTasks, incompleteTasks],
        backgroundColor: ['#10b981', '#ef4444']
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: {
          position: 'bottom',
          labels: {
            font: { size: 11 }
          }
        }
      }
    }
  });

  // カテゴリ別完了率
  const categoryStats = {};
  tasks.forEach(task => {
    if (!task.category) return;
    if (!categoryStats[task.category]) {
      categoryStats[task.category] = { total: 0, completed: 0 };
    }
    categoryStats[task.category].total++;
    if (task.completed) {
      categoryStats[task.category].completed++;
    }
  });

  const categoryLabels = Object.keys(categoryStats);
  const categoryData = categoryLabels.map(cat => {
    const stats = categoryStats[cat];
    return stats.total > 0 ? Math.round((stats.completed / stats.total) * 100) : 0;
  });
  const categoryColors = categoryLabels.map(cat => {
    const category = categories.find(c => c.name === cat);
    return category ? category.color : '#3b82f6';
  });

  new Chart(document.getElementById('category-chart'), {
    type: 'bar',
    data: {
      labels: categoryLabels,
      datasets: [{
        label: '完了率',
        data: categoryData,
        backgroundColor: categoryColors
      }]
    },
    options: {
      responsive: true,
      scales: {
        y: {
          beginAtZero: true,
          max: 100
        }
      },
      plugins: {
        legend: {
          display: false
        }
      }
    }
  });
}

// ========== ユーティリティ関数 ==========

// 空状態を描画
function renderEmptyState() {
  let message = 'タスクがありません';
  let hint = '上のフォームからタスクを追加してください';

  // ステータスフィルター適用中のメッセージ
  if (currentStatusFilter === 'incomplete') {
    message = '未完了のタスクはありません';
    hint = 'すべて完了しています！';
  } else if (currentStatusFilter === 'completed') {
    message = '完了したタスクはありません';
    hint = 'タスクを完了させてみましょう';
  }

  taskListEl.innerHTML = `
    <div class="empty-state">
      <div class="empty-state-icon">📝</div>
      <div class="empty-state-text">${escapeHtml(message)}</div>
      <div class="empty-state-hint">${escapeHtml(hint)}</div>
    </div>
  `;
}

// エラー状態を描画
function renderErrorState(errorMessage) {
  taskListEl.innerHTML = `
    <div class="error-state">
      <div class="error-state-icon">⚠️</div>
      <div class="error-state-text">エラーが発生しました</div>
      <div class="error-state-message">${escapeHtml(errorMessage)}</div>
    </div>
  `;
}

// HTMLエスケープ
function escapeHtml(text) {
  if (text === null || text === undefined) return '';
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}
