/**
 * ドラッグ&ドロップユーティリティ
 *
 * 機能:
 * - HTML5ネイティブDrag & Drop API
 * - タスクの並び替え
 * - 順序の保存
 */

let draggedTaskId = null;
let draggedElement = null;

/**
 * ドラッグ&ドロップをセットアップ
 * @param {HTMLElement} taskListElement - タスクリストの要素
 * @param {Function} onDropCallback - ドロップ時のコールバック (newOrder) => Promise<void>
 */
export function setupDragAndDrop(taskListElement, onDropCallback) {
  if (!taskListElement) {
    console.error('taskListElement is required');
    return;
  }

  // ドラッグ開始
  taskListElement.addEventListener('dragstart', (e) => {
    const taskItem = e.target.closest('.task-item');

    if (!taskItem) return;

    draggedTaskId = taskItem.dataset.taskId;
    draggedElement = taskItem;

    // ドラッグ中の視覚効果
    taskItem.classList.add('dragging');

    // ドラッグデータを設定
    e.dataTransfer.effectAllowed = 'move';
    e.dataTransfer.setData('text/html', taskItem.innerHTML);
  });

  // ドラッグ中
  taskListElement.addEventListener('dragover', (e) => {
    e.preventDefault(); // ドロップを許可

    const taskItem = e.target.closest('.task-item');

    if (!taskItem || taskItem.dataset.taskId === draggedTaskId) return;

    // ドロップ位置を計算
    const rect = taskItem.getBoundingClientRect();
    const midpoint = rect.top + rect.height / 2;

    if (e.clientY < midpoint) {
      // 上に挿入
      taskListElement.insertBefore(draggedElement, taskItem);
    } else {
      // 下に挿入
      taskListElement.insertBefore(draggedElement, taskItem.nextSibling);
    }
  });

  // ドロップ
  taskListElement.addEventListener('drop', async (e) => {
    e.preventDefault();

    const taskItem = e.target.closest('.task-item');
    if (!taskItem) return;

    // 新しい順序を計算
    const newOrder = Array.from(taskListElement.children)
      .map((item, index) => ({
        id: item.dataset.taskId,
        order: index
      }));

    // コールバックで順序を保存
    if (onDropCallback) {
      try {
        await onDropCallback(newOrder);
      } catch (error) {
        console.error('Failed to save order:', error);
      }
    }
  });

  // ドラッグ終了
  taskListElement.addEventListener('dragend', (e) => {
    const taskItem = e.target.closest('.task-item');

    if (taskItem) {
      taskItem.classList.remove('dragging');
    }

    draggedTaskId = null;
    draggedElement = null;
  });
}

/**
 * タスクアイテムをドラッグ可能にする
 * @param {HTMLElement} taskItemElement - タスクアイテムの要素
 */
export function makeDraggable(taskItemElement) {
  if (!taskItemElement) return;

  taskItemElement.setAttribute('draggable', 'true');

  // ドラッグハンドルがなければ追加
  if (!taskItemElement.querySelector('.task-drag-handle')) {
    const taskHeader = taskItemElement.querySelector('.task-header');
    if (taskHeader) {
      const dragHandle = document.createElement('div');
      dragHandle.className = 'task-drag-handle';
      dragHandle.textContent = '⋮⋮';
      dragHandle.title = 'ドラッグして並び替え';

      // task-headerの先頭に挿入
      taskHeader.insertBefore(dragHandle, taskHeader.firstChild);
    }
  }
}

/**
 * 複数のタスクアイテムをドラッグ可能にする
 * @param {NodeList} taskItemElements - タスクアイテムの要素リスト
 */
export function makeAllDraggable(taskItemElements) {
  if (!taskItemElements) return;

  taskItemElements.forEach(taskItemElement => {
    makeDraggable(taskItemElement);
  });
}
